# Copyright 1999-2004, Fred Steinberg, Brown Bear Software

package EventEditForm;
use strict;

use CGI (':standard');
use Calendar::GetHTML;
use Calendar::Date;
use Calendar::Javascript;

# Forms with text areas, etc., for editing events.
# Pass:
#    operation
#    ref to param hash, with 'newOrEdit', 'date', etc. keys
#    repeatStuffP - whether or not to display the items for repeating events 
sub eventEdit {
    my $class = shift;
    my ($operation, $paramHashRef) = @_;

    my $prefs   = $operation->prefs;
    my $i18n    = $operation->I18N;
    my $calName = $operation->calendarName;
    my $username = $operation->getUsername;

    my $event      = $paramHashRef->{event}; # not defined for a *new* event
    my $date       = $paramHashRef->{date};
    my $allOrOne   = $paramHashRef->{allOrOne}   || '';
    my $mainHeader = $paramHashRef->{mainHeader} || '';
    my $newOrEdit  = $paramHashRef->{newOrEdit}  || 'new';
    my $displayCal = $paramHashRef->{displayCal} || $calName;
    my $viewCal    = $paramHashRef->{viewCal}; # for getting back to planner
    my $calendarList = $paramHashRef->{calList}; # for multi-add

    my $eventID   = $event ? $event->id()        : undef;
    my $eventText = $event ? $event->text()      : '';
    my $export    = $event ? $event->export : $prefs->EventPrivacy || 'Public';
    my $popupText = $event ? ($event->link() || $event->popup() || '') : '';
    my $timePeriod = $event ? $event->timePeriod
                            : ($prefs->DefaultTimePeriod || undef);

    my $mailTo    = $event ? $event->mailTo   : '';
    my $mailCC    = $event ? $event->mailCC   : '';
    my $mailBCC   = $event ? $event->mailBCC  : '';
    my $mailText  = $event ? $event->mailText : '';
    my $defNotify = $prefs->DefaultSubsNotify;
    my $reminderTo;
    my $reminderTimes = $event ? ($event->reminderTimes || '') : '';
    my @reminderTimes = split /\s/, $reminderTimes;

    if ($event) {
        $reminderTo = $event->reminderTo;
    } else {
        my $user = User->getUser ($username);
        $reminderTo = $user->email if $user;
    }

    my $drawBorder = $event ? $event->drawBorder() : $prefs->DefaultBorder;
    my ($primaryCat, @moreCats) = $event ? $event->getCategoryList
                                         : ($prefs->DefaultCategory || undef);
    my $bgColor    = ($event && $event->bgColor())  || 'Default';
    my $fgColor    = ($event && $event->fgColor())  || 'Default';

    my $prompts = $prefs->EditFormPrompts || '';
    my %prompts = split ' ;; ', $prompts;

    # Since we can be called for a new event or to edit an existing event,
    # we've got to see if we've already got repeat info
    my $repeatInfo = defined ($event) ? $event->repeatInfo() : undef;

    my ($action, $buttonText, $nextOp, $textPrompt, $mailPrompt,
        $copyButtonText);
    if (lc ($newOrEdit) eq 'new') {
        $action     = 'EventNew';
        $nextOp     = 'ShowDay';
        $buttonText = $i18n->get ('Create Event');
        $textPrompt = $prompts{TextNew} ||
                          $i18n->get ('Enter text for a new event:');
        $mailPrompt = $i18n->get ('Specify email addresses to notify ' .
                                  'that this event has been added.');
    } else {    # editing existing
        $action     = 'EventReplace';
        $nextOp     = 'ShowDay';
        $buttonText = $i18n->get ('Replace Event');
        $textPrompt = $prompts{TextEdit} ||
                          $i18n->get ('Modify the text for this event:');
        $mailPrompt = $i18n->get ('Add or change email addresses to notify ' .
                                  'that this event has been modified.');
        $copyButtonText = $i18n->get ('Copy Event')
            unless ($repeatInfo and $allOrOne !~ /all/i);
    }

    my $frequency  = $repeatInfo ? $repeatInfo->frequency() : '';
    my $period     = $repeatInfo ? $repeatInfo->period()    : '';
    my $startDate  = ($repeatInfo and lc ($allOrOne) ne 'only')
                                 ? $repeatInfo->startDate() : Date->new($date);
    my $endDate      = $repeatInfo ? $repeatInfo->endDate()      : '';
    my $monthWeek    = $repeatInfo ? $repeatInfo->monthWeek()    : '';
    my $monthMonth   = $repeatInfo ? $repeatInfo->monthMonth()   : '';
    my $skipWeekends = $repeatInfo ? $repeatInfo->skipWeekends() : '';

    my ($startHour, $startMinute, $endHour, $endMinute);
    my $displayDate;

    if ($event and defined $event->startTime) {
        # Adjust for timezones
        my $zoffset = $prefs->Timezone || 0;
        if ($zoffset) {
            $startDate = $event->getDisplayDate ($startDate, $zoffset);
            $displayDate = $startDate;
        }
        my ($start, $end) = $event->getDisplayTime ($zoffset);

        $startHour   = int ($start / 60);
        $startMinute = $start % 60;
        if (defined $end) {
            $endHour   = int ($end / 60);
            $endMinute = $end % 60;
        }
    } else {
        if (my $start = $paramHashRef->{defaultStartTime}) {
            $startHour = int ($start / 60);
            $startMinute = $start % 60;
        }
        if (my $end = $paramHashRef->{defaultEndTime}) {
            $endHour = int ($end / 60);
            $endMinute = $end % 60;
        }
    }

    my $defaultRepeat = 'None';
    if ($repeatInfo) {
        if ($period) {
            $period = join (' ', sort @$period) if (ref ($period));
            $defaultRepeat = 'Repeat';
        } else {
            $defaultRepeat = 'ByWeek';
        }
        $monthWeek = join ' ', @$monthWeek if (ref ($monthWeek));
    }

    # Start time, end time stuff
    $startHour = -1 if !defined $startHour;
    $endHour   = -1 if !defined $endHour;
    $startMinute = ($startHour >= 0) ? $startMinute || 0 : -1;
    $endMinute   = ($endHour   >= 0) ? $endMinute   || 0 : -1;

    my $milTime = $prefs->MilitaryTime;
    my $none18 = $i18n->get ('None');
    my ($startTimeHourPopup,
        @startTimeRadio) = $class->_hourPopup ('nameBase'     => 'StartHour',
                                               'default'      => $startHour,
                                               'militaryTime' => $milTime,
                                               'None'         => $none18);
    my ($endTimeHourPopup,
        @endTimeRadio)  = $class->_hourPopup ('nameBase'      => 'EndHour',
                                              'default'      => $endHour,
                                              'militaryTime' => $milTime,
                                              'None'         => $none18);

    # First thing, lets add the Javascript we need
    my $html = $class->_setMinutesPopup ();

    # Form for adding new event string
    my $addOrEditURL = $operation->makeURL ({Op => $action});

    $html .= startform ({-action => $addOrEditURL,
                         -name => 'EventEditForm'});

    $html .= GetHTML->SectionHeader ($mainHeader);

    my $hideThese = $prefs->EditFormHide || '';
    my %hideIt = (whenInc     => ($hideThese =~ /whenInc/i)     || 0,
                  border      => ($hideThese =~ /border/i)      || 0,
                  colors      => ($hideThese =~ /colors/i)      || 0,
                  category    => ($hideThese =~ /category/i)    || 0,
                  moreCats    => ($hideThese =~ /moreCats/i)    || 0,
                  details     => ($hideThese =~ /details/i)     || 0,
                  repeat      => ($hideThese =~ /repeat/i)      || 0,
                  subscribers => ($hideThese =~ /subscribers/i) || 0,
                  mail        => ($hideThese =~ /mail/i)        || 0);

    my $dateTable = table ({-cellspacing => 0, -cellpadding => 0},
                           Tr (td (b (($repeatInfo and ($allOrOne =~ /all/i))
                                      ? $i18n->get ('Start Date:')
                                      : $i18n->get ('Date:')))),
                           Tr (td (GetHTML->datePopup ($i18n,
                                                  {name    => 'Date',
                                                   start   => $startDate - 750,
                                                   default => $startDate,
                                                   op      => $operation}))));

    my $exportTable = '';
    unless ($hideIt{whenInc}) {
        $exportTable = table ({-cellspacing => 0, -cellpadding => 0},
                             Tr (td (b
                                     ($i18n->get
                                      ('When included in other calendars:')))),
                             Tr (td (popup_menu (-name => 'ExportPopup',
                                                 -default => lc ($export),
                                                 -Values  => ['public',
                                                              'private',
                                                              'nopopup',
                                                              'unavailable',
                                                              'outofoffice'],
                                                 -labels =>
                          {public  => $i18n->get ("Display this event"),
                           private => $i18n->get ("Don't display this event"),
                           nopopup => $i18n->get
                                         ("Display event text, but not Popup"),
                           unavailable =>$i18n->get ("Display 'Unavailable'"),
                           outofoffice =>$i18n->get
                                         ("Display 'Out of Office'"),
                          }))));
    }

    my $textTable = table (Tr (td (b ($textPrompt))),
                           Tr (td (textarea (-name    => 'EventText',
                                             -rows    => 2,
                                             -columns => 35,
                                             -default => "$eventText",
                                             -wrap    => 'SOFT'))));

    my $whichTime = $prefs->TimeEditWhich || 'startend';
    my $timePopups;
    if ($whichTime =~ /^(startend|both)$/i) {
        $timePopups = table (Tr (td ({-align => 'RIGHT'},
                                    $i18n->get('Start Time:')),
                                td ({-align => 'RIGHT'}, $startTimeHourPopup),
                                td ({-align => 'LEFT'},
                                    $class->_minutePopup ('name'    =>
                                                          'StartMinutePopup',
                                                          'default' =>
                                                          $startMinute)),
                                td (@startTimeRadio)),
                            Tr (td ({-align => 'RIGHT'},
                                    $i18n->get ('End Time:')),
                                td ({-align => 'RIGHT'}, $endTimeHourPopup),
                                td ({'-align' => 'LEFT'},
                                    $class->_minutePopup ('name'    =>
                                                          'EndMinutePopup',
                                                          'default' =>
                                                          $endMinute)),
                                td (@endTimeRadio)));
    }

    my $periodTable;
    if ($whichTime =~ /^(period|both)$/i) {
        my $periods = $prefs->getTimePeriods ('inherit'); # master too
        # Sort on start time; periods are IDs labels are names
        my @timePeriods = sort {$periods->{$a}->[1] <=> $periods->{$b}->[1]}
                             keys %$periods;
        my %labels = map {$_ => $periods->{$_}->[0]} @timePeriods;
        unshift @timePeriods, '-';
        $periodTable = table (Tr (td (b ($i18n->get ('Time Period') . ': ')),
                                  td (popup_menu (-name     => 'TimePeriod',
                                                  -default  => $timePeriod,
                                        -onChange => 'setTimeFromPeriod(this)',
                                                  -values   => \@timePeriods,
                                                  -labels   => \%labels))));
        # Add the Javascript we need
        my $tzoffset = $prefs->Timezone || 0;
        $html .= $class->_setTimesFromPeriodJS ($milTime, $tzoffset,
                                                map {$periods->{$_}}
                                                    @timePeriods);
    }

    my $timeTable;
    if ($periodTable and $timePopups) {
        $timeTable = "<br><table border=1><tr>" .
                     "<td align='center'>$periodTable<hr width=\"80%\">" .
                     "$timePopups</td></tr></table><br>";
    } else {
        $timeTable = $periodTable || $timePopups || '&nbsp;';
    }

    my $border = '';
    unless ($hideIt{border}) {
        $border = checkbox (-name    => 'BorderCheckbox',
                            -checked => $drawBorder,
                            -label   => '');
    }

    my $colorNames = '';
    unless ($hideIt{colors}) {
        $html .= Javascript->ColorPalette ($operation);
        $colorNames = a ({-href   => "Javascript:ColorWindow()"},
                         $i18n->get ('Color Names'));
    }

    my ($categoryPopup, $moreCategories) = ('', '');
    unless ($hideIt{category}) {
        my $catObjs = $prefs->getCategories (1);

        my $onClick = 'setStyle(this)';
        my ($bgs, $fgs);
        ($categoryPopup, $bgs, $fgs) = _makeCatPopup (name     => 'Category',
                                                      cats     => $catObjs,
                                                     selected => [$primaryCat],
                                                      onClick  => $onClick);
        my $jsBG = join ',', @$bgs;
        my $jsFG = join ',', @$fgs;

        unless ($hideIt{moreCats}) {
            $moreCategories =
                scrolling_list (-name     => 'MoreCategories',
                                -default  => [@moreCats],
                                -Values   => [sort {lc($a) cmp lc($b)}
                                                            keys %$catObjs],
                                -size     => 5,
                                -multiple => 'true');
        }

        $html .= <<END_SCRIPT;
 <script language="JavaScript">
 <!--
    function setStyle (theList) {
        if (navigator.appName.toLowerCase().indexOf("microsoft") > -1) {
            return;
        }
        var bgColors = new Array ($jsBG);
        var fgColors = new Array ($jsFG);
        theList.style.backgroundColor = bgColors[theList.selectedIndex];
        theList.style.color           = fgColors[theList.selectedIndex];
    }
-->
 </script>
END_SCRIPT
    }

    my %labels = (category => $prompts{Category} || $i18n->get ('Category'),
                  moreCats => $prompts{MoreCats} ||
                                             $i18n->get ('More Categories'),
                  backg    => $i18n->get ('Background'),
                  foreg    => $i18n->get ('Foreground'),
                  border   => $i18n->get ('Draw Border'));
    foreach (qw/category moreCats border/) {
        $labels{$_} = '' if $hideIt{$_};
    }
    $labels{backg} = $labels{foreg} = '' if $hideIt{colors};
    $labels{moreCats} = '' if $hideIt{category};

    my ($bgField, $fgField) = ('', '');
    unless ($hideIt{colors}) {
        $bgField = textfield (-name      => 'BackgroundColor',
                              -default   => $bgColor,
                              -size      => 10,
                              -maxlength => 20);
        $fgField = textfield (-name      => 'ForegroundColor',
                              -default   => $fgColor,
                              -size      => 10,
                              -maxlength => 20);
    }

    my $miscStuff = table ({-width => '100%',
                            -cellspacing => 0, -cellpadding => 0},
                           Tr (th [@labels{qw/category moreCats backg
                                              foreg border/}]),
                           Tr (td ({-align => 'center',
                                    -valign => 'top'},
                                   $categoryPopup),
                               td ({-align => 'center'},
                                   $moreCategories),
                               td ({-align => 'center', valign => 'top'},
                                   $bgField),
                               td ({-align => 'center', valign => 'top'},
                                   $fgField),
                               td ({-align => 'center', valign => 'top'},
                                   $border),
                               td ($colorNames)));

    my $popupTable = '';
    unless ($hideIt{details}) {
        my $prompt = $prompts{Details} ||
                       $i18n->get ('Enter a URL, or text for a popup window:');
        $popupTable = table
            (Tr (td (b ($prompt))),
             Tr (td (textarea (-name    => 'PopupText',
                               -rows    => 4,
                               -cols    => 60,
                               -default => "$popupText",
                               -wrap    => 'SOFT'))),
             Tr (td ($prompts{SubDetails} ||
                     $i18n->get ('Anything starting with '    .
                                 'http:, https:, mailto:, '    .
                                 'ftp:, file:, or a \'www\' '  .
                                 'string (e.g. '               .
                                 '<i>www.domainname.com</i>) ' .
                                 'will be a link. Anything '   .
                                 'else will be popup text.'))));
    }

    # Allow specifying which calendars to add the event to. So, we need a
    # list of all calendars this user has Add permission in.
    my $selectCalRow = '';
    my $whichUsers = $prefs->MultiAddUsers;
    my $showMulti  = (($whichUsers  =~ /anyone/i)
                     or
                      ($whichUsers =~ /caladmin/i and
                       $operation->permission->permitted ($username, 'Admin'))
                     or
                      ($whichUsers =~ /sysadmin/i and
                       Permissions->new (MasterDB->new)->permitted ($username,
                                                                    'Admin')));
    # Don't show if we're already specifying multiple cals,
    # or if we're editing something other then reg. event or entire series
    my $multiCalDisplayed;
    if (!defined $calendarList and $showMulti and ($allOrOne =~ /all/i)) {
        my $whichCals = $prefs->MultiAddCals || 'permitted';
        my %allowed = map {$_ => 1}
                       grep {Permissions->new (Database->new ($_))->permitted
                                                            ($username, 'Add')}
                       MasterDB->getAllCalendars;
        my @myCals = ();
        if ($whichCals =~ /ingroup/i) {
            my @groups = $prefs->getGroups;
            my ($groups, $noGroup) = MasterDB->getCalendarsInGroup (@groups);
            @myCals = (@$groups, @$noGroup);
        } elsif ($whichCals =~ /included/i) {
            @myCals = $prefs->getIncludedCalendarNames;
        } elsif ($whichCals =~ /permitted/i) {
            @myCals = keys %allowed;
        }
        my @theCals;
        foreach (@myCals) {
            next unless $allowed{$_};
            next if ($_ eq $calName);
            push @theCals, $_;
        }
        # Perl 5.8.4 bug - does not like @foo = ('baz', sort @foo);
#        @theCals = ($calName, sort {lc($a) cmp lc($b)} @theCals);
#        if (@theCals > 1) {
        my @xxxtheCals = ($calName, sort {lc($a) cmp lc($b)} @theCals);
        if (@xxxtheCals > 1) {
            $multiCalDisplayed = 1;
            my $calPopup = scrolling_list (-name     => 'WhichCalendars',
                                           -default  => $calName,
#                                           -Values   => \@theCals,
                                           -Values   => \@xxxtheCals,
                                           -size     => 5,
                                           -multiple => 'true');
            my ($prompt, $message);
            if (lc ($newOrEdit) eq 'new') {
                $prompt  = $i18n->get ('Add to Which Calendars?');
                $message = '<small><b>Note:</b> the event will not be ' .
                           'added to the current calendar unless it is ' .
                           'selected.</small>';
            } else {
                $prompt  = $i18n->get ('Copy to Which Calendars?');
                $message = '<b>Note:</b> If other calendars are selected, ' .
                           'the event will be copied to those calendars, '  .
                           'even if you select "Replace"';
                $message = '<b>Note:</b> This selection is only used when ' .
                           '<b>copying</b> ' . 'events to other calendars.';
            }

            my $table = table (Tr (td [$prompt, '&nbsp;',
                                       $calPopup, $message]));
            $selectCalRow = Tr (td ({-colspan => 2}, $table));
        }
    }

    $html .= table ({-class => 'EntryWidgets',
                     -width => '100%'},
                    Tr (td ([$dateTable, $exportTable])),
                    Tr (td ([$textTable, $timeTable])),
                    Tr (td ({-colspan => 2}, $miscStuff)),
                    Tr (td ({-colspan => 2}, $popupTable)),
                    $selectCalRow);

    $html .= '<br>';
    $html .= _submitButtons ($buttonText, $copyButtonText, $paramHashRef);

    # Display repeat stuff unless prefs tell us not to or we're editing a
    # single instance of a repeating event
    unless ($hideIt{repeat} or
            $repeatInfo and $allOrOne !~ /all/i) {

        $html .= '<br><hr>';

        my @repeatRadios = radio_group ('-name'      => 'RepeatRadio',
                                        '-values'    => ['None', 'Repeat',
                                                         'ByWeek'],
                                        '-default'   => $defaultRepeat,
                                        '-labels'    =>
                                             {'None'   => ' ' .
                                                   $i18n->get("Don't Repeat"),
                                              'Repeat' => ' ' .
                                                   $i18n->get('Repeat') . ' ',
                                              'ByWeek' => ' ' .
                                           $i18n->get('Repeat on the') . ' '});

        my %nthLabels = (1  => '',
                         2  => ' Other',
                         3  => ' Third',
                         4  => ' Fourth',
                         5  => ' Fifth',
                         6  => ' Sixth',
                         7  => ' Seventh',
                         8  => ' Eighth',
                         9  => ' Ninth',
                         10 => ' Tenth',
                         11 => ' Eleventh',
                         12 => ' Twelfth',
                         13 => ' Thirteenth',
                         14 => ' Fourteenth',
                         15 => ' Fifteenth',
                         16 => ' Sixteenth',
                         17 => ' Seventeenth',
                         18 => ' Eighteenth',
                         19 => ' Nineteenth',
                         20 => ' Twentieth');
        %nthLabels = map {$_ => $i18n->get ("Every$nthLabels{$_}")}
                         keys %nthLabels;

        my $freqPopup = popup_menu ('-name'    => 'Frequency',
                                    '-default' => $frequency,
                                    '-values'  => [1..20],
                                    '-labels'  => \%nthLabels,
                                    '-onChange'=>
                                    'this.form.RepeatRadio[1].checked = true');

        my @values = ('day', 'dayBanner', 'week', 'month', 'year',
                      '1 2 3 4 5', '1 2 3 4 5 6', '1 3 5', '1 3', '2 4',
                      '6 7', '2 5', '5 6', '4 5 6', '1 2 3 4', '2 3 4 5');
        my %labels = ('day'       => $i18n->get ('Day'),
                      'dayBanner' => $i18n->get ('Day (Bannered)'),
                      'week'      => $i18n->get ('Week'),
                      'month'     => $i18n->get ('Month'),
                      'year'      => $i18n->get ('Year'),
                      '1 2 3 4 5' => $i18n->get ('Monday')    . ' - ' .
                                     $i18n->get ('Friday'),
                      '1 2 3 4 5 6' => $i18n->get ('Monday')    . ' - ' .
                                       $i18n->get ('Saturday'),
                      '1 3 5'     => $i18n->get ('Monday')    . ', ' .
                                     $i18n->get ('Wednesday') . ', ' .
                                     $i18n->get ('Friday'),
                      '1 3'       => $i18n->get ('Monday')   . ', ' .
                                     $i18n->get ('Wednesday'),
                      '2 4'       => $i18n->get ('Tuesday')   . ', ' .
                                     $i18n->get ('Thursday'),
                      '2 5'       => $i18n->get ('Tuesday')   . ', ' .
                                     $i18n->get ('Friday'),
                      '1 2 3 4'   => $i18n->get ('Monday')    . ' - ' .
                                     $i18n->get ('Thursday'),
                      '2 3 4 5'   => $i18n->get ('Tuesday') . ' - ' .
                                     $i18n->get ('Friday'),
                      '5 6'       => $i18n->get ('Friday')   . ', ' .
                                     $i18n->get ('Saturday'),
                      '4 5 6'     => $i18n->get ('Thursday')    . ' - ' .
                                     $i18n->get ('Saturday'),
                      '6 7'       => $i18n->get ('Saturday')  . ', ' .
                                     $i18n->get ('Sunday'));

        my $def = $period || $prefs->DefaultPeriod;
        my $periodPopup .= popup_menu ('-name'    => 'Period',
                                       '-default' => $def,
                                       '-values'  => \@values,
                                       '-labels'  => \%labels,
                                       '-onChange'=> 
                                    'this.form.RepeatRadio[1].checked = true');

        @values = ('1', '2', '3', '4', '5', '6', '1 3', '2 4', '1 5');
        %labels = ('1'   => $i18n->get ('First'),
                   '2'   => $i18n->get ('Second'),
                   '3'   => $i18n->get ('Third'),
                   '4'   => $i18n->get ('Fourth'),
                   '5'   => $i18n->get ('Last'),
                   '6'   => $i18n->get ('Fifth, if it exists'),
                   '1 3' => $i18n->get ('First and Third'),
                   '2 4' => $i18n->get ('Second and Fourth'),
                   '1 5' => $i18n->get ('First and Last'));

        my $mwPopup = popup_menu ('-name'    => 'MonthWeek',
                                  '-default' => $monthWeek,
                                  '-values'  => \@values,
                                  '-labels'  => \%labels,
                                  '-onChange'=> 
                                    'this.form.RepeatRadio[2].checked = true');

        my $dayName = b ($i18n->get ($startDate->dayName)) . ' '
                      . $i18n->get ('every');
        my $month18 = lc ($i18n->get ('Month'));
        my $mmPopup = popup_menu ('-name'    => 'MonthMonth',
                                  '-default' => $monthMonth,
                                  '-values'  => [ 1, 2, 3, 4, 5, 6, 12],
                                  '-labels'  => {'1' => $month18,
                                                 '2' => $i18n->get ('other') .
                                                        " $month18",
                                                 '3' => $i18n->get ('third') .
                                                        " $month18",
                                                 '4' => $i18n->get('fourth') .
                                                        " $month18",
                                                 '5' => $i18n->get('fifth') .
                                                        " $month18",
                                                 '6' => $i18n->get('sixth') .
                                                        " $month18",
                                                 '12' => $i18n->get ('year')},
                                  '-onChange'=> 
                                    'this.form.RepeatRadio[2].checked = true');

        $html .= GetHTML->SectionHeader ($i18n->get ('Repeat Information') .
                                         '&nbsp;<i>(' .
                                         $i18n->get ('Optional') .
                                         '</i>)');

#        '&nbsp;&nbsp;&nbsp;<small>[<a href="javascript:hideFoo();">Hide/Show</a>]</small>'

        $html .= '<div id="RepeatSection">';
#         $html .= qq [<script language="Javascript">
#             var isHidden = false;
#             function hideFoo (){
#                isHidden = !isHidden;
#                var el = document.getElementById("RepeatSection");
#                if (isHidden) {
#                    el.style.display = "none";
#                } else {
#                    el.style.display = "block";
#                }
#             }</script>];

        # Put the three repeat options (none, every, by nth day of month)
        # in a table, so they have a background
        my $repeatTable =
                table ({-cellspacing  => 0, -width => '100%'},
                       Tr (td ($repeatRadios[0])),
                       Tr (td ($repeatRadios[1] . $freqPopup . $periodPopup),
                           td ({-align => 'right'},
                               checkbox (-name    => 'SkipWeekends',
                                         -checked => $skipWeekends,
                                         -label   => ' ' .
                                             $i18n->get ("Skip Weekends")))),
                       Tr (td ({-colspan => 2},
                               $repeatRadios[2] . $mwPopup   . $dayName .
                               $mmPopup)));

        # Now do the Repeat Until Stuff
        my $forever = $i18n->get ('Forever!');
        my @radio = radio_group ('-name'    => 'RepeatUntilRadio',
                                 '-values'  => [' ', " $forever"],
                                 '-default' => (!$endDate or
                                                  $endDate != Date->openFuture)
                                               ? ' ' : " $forever");

        my $untilTable =
                table ({-cellspacing => 0},
                       Tr (td ({-rowspan => 2,
                                -valign  => 'center'},
                               '<b>' . $i18n->get ('Repeat until') .
                               '</b>&nbsp; &nbsp;'),
                           td ($radio[0] .
                               GetHTML->datePopup ($i18n,
                                                   {name    => 'Until',
                                                    start   => $date,
                                                    op      => $operation,
                                                    default => $endDate ||
                                                                     $date + 1,
                                                    onChange =>
                           'this.form.RepeatUntilRadio[0].checked = true;' .
                           'if (this.form.RepeatRadio[0].checked == true) {'.
                           '  this.form.RepeatRadio[1].checked = true}'}))),
                       Tr (td ($radio[1])));

        $html .= table ({-class => 'EntryWidgets',
                         -border => 0, -cellspacing => 0,
                         -width   => '100%'},
                        Tr ([td ($repeatTable),
                             td ('<hr width="25%">'),
                             td ($untilTable)]));

        $html .= '<br>';
        $html .= _submitButtons ($buttonText, $copyButtonText, $paramHashRef);
        $html .= '</div>';
    }

    # Display current subscribers, if editing an event which has some
    if ($event and Defines->mailEnabled and
        my $subscribers = $event->getSubscribers ($calName) and
        !$hideIt{subscribers}) {
        $html .= '<br><hr>';
        $html .= GetHTML->SectionHeader ($i18n->get ('Subscriptions'));

        # Sort them; rejoin with spaces so it wraps in textarea
        my @subs = split /,/, $subscribers;
        $subscribers = join ', ', sort {lc($a) cmp lc ($b)} @subs;

        $html .= table ({-class => 'EntryWidgets',
                         -width => '100%'},
                        Tr (td ({-align => 'right'},
                                '<small>' .
                                $i18n->get ('Email addresses subscribed to ' .
                                            'this event:') .
                                '</small>'),
                            td ({-align => 'left'},
                                textarea (-name    => 'SubscriberAddresses',
                                          -rows    => 2,
                                          -cols    => 60,
                                          -default => $subscribers,
                                          -wrap    => 'SOFT'))));
    }

    # JS cleanup is always called from ShowDay
    $html .=  <<END_SCRIPT;
<script language="Javascript">
<!-- start
    var EmailSelectionPopupWindow;
    function cleanUpPopups () {
        if (EmailSelectionPopupWindow)
            EmailSelectionPopupWindow.close();
    }
 -->
</script>
END_SCRIPT


    # Display mail stuff unless prefs tell us not to.
    unless ($hideIt{mail}) {

        $html .= "\n<br><hr>\n";
        $html .= GetHTML->SectionHeader ($i18n->get ('Email Notification') .
                                         '&nbsp;<i>(' .
                                         $i18n->get ('Optional') . '</i>)');

        my $showAddrSelect = lc ($prefs->EmailSelector || '') ne 'none';
        if ($showAddrSelect) {
            my $width  = $prefs->EmailSelectPopupWidth  || 400;
            my $height = $prefs->EmailSelectPopupHeight || 300;
            $html .= Javascript->MakePopupFunction
                       ($operation->makeURL ({Op => 'EmailSelector'}),
                        'EmailSelection', $width, $height);
        }

        my $addressTable =
            table ({-width => '100%', -cellspacing => 0 ,-cellpadding => 0},
                   Tr (td ({-colspan => 3}, '<small>' . $mailPrompt . ' ' .
                           $i18n->get ('(Use commas between addresses.)') .
                           '</small>')),
                   Tr (td ({-align => 'right', -width => '10%'},
                           $i18n->get ('TO:')),
                       td (textfield (-name    => 'MailTo',
                                      -default => $mailTo,
                                      -size    => 40)),
                       $showAddrSelect ?
                           td (a ({-href =>'JavaScript:EmailSelectionPopup()'},
                                  $i18n->get ('Email Address Selector')))
                           : ''),
                   Tr (td ({-align => 'right'}, $i18n->get ('CC:')),
                       td (textfield (-name    => 'MailCC',
                                      -default => $mailCC,
                                      -size    => 40))),
                   Tr (td ({-align => 'right'}, $i18n->get ('BCC:')),
                       td (textfield (-name    => 'MailBCC',
                                      -default => $mailBCC,
                                      -size    => 40)),
                       Defines->mailEnabled ?
                           td (checkbox (-name    => 'NotifySubscribers',
                                         -checked => $defNotify,
                                         -label   =>
                                         $i18n->get ('Automatically notify ' .
                                                     'calendar subscribers')))
                            : ''));
        my $commentTable =
            table ({-width => '100%', -cellspacing => 0 ,-cellpadding => 0},
                   Tr (td ({-colspan => 2}, '<small>' .
                           $i18n->get ('Specify any additional comments '   .
                                       'that you would like included with ' .
                                       'the notification.') . '</small>')),
                   Tr (td ({-width => '10%'}, '&nbsp;'),
                       td (textarea (-name    => 'MailComments',
                                     -rows    => 2,
                                     -cols    => 60,
                                     -default => $mailText,
                                     -wrap    => 'SOFT'))));

        $html .= table ({-class => 'EntryWidgets',
                         -width => '100%'},
                        Tr ([td ($addressTable), td ($commentTable)]));

        if (Defines->mailEnabled) {
            $html .= '<br>';
            $html .= GetHTML->SectionHeader ($i18n->get ('Email Reminders') .
                                             '&nbsp;<i>(' .
                                            $i18n->get ('Optional') . '</i>)');
            my @reminderValues = (0, 5, 10, 15, 20, 30, 45, 60, 120, 180, 360,
                                  720, 1440, 1440*2,  1440*3,  1440*4, 1440*5,
                                  1440*6,    1440*7,  1440*8,  1440*9, 1440*10,
                                  1440*11,   1440*12, 1440*13, 1440*14);
            my ($minute, $minutes) = ($i18n->get ('minute'),
                                      $i18n->get ('minutes'));
            my ($hour, $hours)     = ($i18n->get ('hour'),
                                      $i18n->get ('hours'));
            my ($day, $days)       = ($i18n->get ('day'),
                                      $i18n->get ('days'));
            my ($week, $weeks)     = ($i18n->get ('week'),
                                      $i18n->get ('weeks'));
            my ($month, $months)   = ($i18n->get ('month'),
                                      $i18n->get ('months'));
            my %reminderLabels = (0       => '----',
                                  5       => "5  $minutes",
                                  10      => "10 $minutes",
                                  15      => "15 $minutes",
                                  20      => "20 $minutes",
                                  30      => "30 $minutes",
                                  45      => "45 $minutes",
                                  60      => "1 $hour",
                                  120     => "2 $hours",
                                  180     => "3 $hours",
                                  360     => "6 $hours",
                                  720     => "12 $hours",
                                  1440    => "1 $day",
                                  2880    => "2 $days",
                                  1440*3  => "3 $days",
                                  2880*2  => "4 $days",
                                  1440*5  => "5 $days",
                                  1440*6  => "6 $days",
                                  1440*7  => "1 $week",
                                  1440*8  => "8 $days",
                                  1440*9  => "9 $days",
                                  1440*10 => "10 $days",
                                  1440*11 => "11 $days",
                                  1440*12 => "12 $days",
                                  1440*13 => "13 $days",
                                  1440*14 => "2 $weeks",
                                  1440*21 => "3 $weeks",
                                  1440*28 => "4 $weeks",
                                  1440*35 => "5 $weeks");

            foreach (15..30) {
                push @reminderValues, 1440*$_;
                $reminderLabels{1440*$_} = "$_ $days";
            }
            foreach (2..12) {
                push @reminderValues, 1440 * 30 * $_;
                $reminderLabels{1440 * 30 * $_} = "$_ $months";
            }

            my $reminderTimeTable =
                table ({-cols => 6, -cellpadding => 0, -cellspacing => 0},
                       Tr (td ({-nowrap => 1},
                               $i18n->get ('Send reminder email:')),
                           td (popup_menu (-name    => 'MailReminder',
                                           -default => $reminderTimes[0],
                                           -values  => \@reminderValues,
                                           -labels  => \%reminderLabels)),
                           td ({-nowrap => 1},
                               $i18n->get ('before, and')),
                           td (popup_menu (-name    => 'MailReminder2',
                                           -default => $reminderTimes[1],
                                           -values  => \@reminderValues,
                                           -labels  => \%reminderLabels)),
                           td ({-nowrap => 1},
                               $i18n->get ('before the event.')),
                           td ('&nbsp;')),
                       Tr (td ({-align => 'right', -nowrap => 1},
                               $i18n->get ('Email Address:')),
                           td ({-colspan => 5},
                               textfield (-name    => 'ReminderAddress',
                                          -default => $reminderTo,
                                          -size    => 40))));

            $html .= table ({-class => 'EntryWidgets',
                             -width => '100%'},
                            Tr (td ($reminderTimeTable)));
        }
        $html .= '<br>';
        $html .= _submitButtons ($buttonText, $copyButtonText, $paramHashRef);
    }

    # We need to 'override' params, since we call ourself.

    if (defined $calendarList and ref $calendarList) {
        $html .= hidden (-name     => 'WhichCalendars',
                         -override => 1,
                         -value    => $calendarList); # must be listref
        $nextOp = $paramHashRef->{nextOp};
    }
    if ($multiCalDisplayed) {
        $html .= hidden (-name     => 'MultiCalDisplayed',
                         -override => 1,
                         -value    => 1);
    }

    $html .= hidden (-name     => 'CalendarName',
                     -override => 1,
                     -value    => $calName);
    $html .= hidden (-name     => 'Date',
                     -override => 1,
                     -value    => "$date");
    $html .= hidden (-name     => 'DisplayDate',
                     -override => 1,
                     -value    => "$displayDate") if $displayDate;
    $html .= hidden (-name     => 'OldEventID',
                     -override => 1,
                     -value    => $eventID) if defined $eventID;
    $html .= hidden (-name     => 'AllOrOne',
                     -override => 1,
                     -value    => $allOrOne);
    $html .= hidden (-name     => 'Op',
                     -override => 1,
                     -value    => $action);
    $html .= hidden (-name     => 'NextOp',
                     -override => 1,
                     -value    => $nextOp);
    $html .= hidden (-name     => 'DisplayCal',
                     -override => 1,
                     -value    => $displayCal);
    $html .= hidden (-name     => 'ViewCal',
                     -override => 1,
                     -value    => $viewCal) if $viewCal;
    $html .= hidden (-name     => 'FromPopupWindow',
                     -value    => $paramHashRef->{fromPopupWindow})
        if $paramHashRef->{fromPopupWindow};

    $html .= $operation->hiddenDisplaySpecs;

    $html .= endform();

    return qq (<div class="EventEditForm">$html</div>\n);
}

sub _submitButtons {
    my ($submitText, $copyText, $params) = @_;

    my $html .= submit (-name  => 'Submit',
                        -value => $submitText);
    if ($copyText) {
        $html .= ' &nbsp; ';
        $html .= submit (-name  => 'CopyEvent',
                         -value => $copyText);
    }

#    $html .= ' &nbsp; &nbsp; ' . reset () . ' &nbsp; ';

    # If editing, maybe allow deleting. See EventReplace
    if ($params->{showDelete}) {
        $html .= ' &nbsp; ';
        $html .= submit (-name => 'DeleteEvent',
                         -value => $params->{showDelete});
    }

    unless ($params->{noCancel}) {
        $html .= ' &nbsp; ';
        $html .= submit (-name => 'Cancel',
                         -onClick => $params->{cancelOnClick});
    }

    $html;
}

# Produce a popup to select an hour. If using 12 hour time, an am/pm radio
# will also be created. Names will be $nameBase . 'Popup', $nameBase . 'Radio'
# Returns a list of 1 or 2 strings. (Second is the radio, obviously.)
# Pass hash pairs with 'nameBase', 'default' keys.
# (Default should be int, -1 -> 23)
# Defaults to None.
sub _hourPopup {
    my $className = shift;
    my %args = (nameBase     => 'hour',
                default      => -1,
                militaryTime => 1,
                None         => 'None',
                @_);

    $args{default} = -1 if ($args{default} < -1 or $args{default} > 23);

    my ($values, $labels, $popup, @radio);

    if ($args{'militaryTime'}) { # true or false
        $values = [-1, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11,
                      12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23];

        $labels = {'-1' => $args{'None'},
                    '0' => '0',   '1' => '1',   '2' => '2',   '3' => '3',
                    '4' => '4',   '5' => '5',   '6' => '6',   '7' => '7',
                    '8' => '8',   '9' => '9',  '10' => '10', '11' => '11',
                   '12' => '12', '13' => '13', '14' => '14', '15' => '15',
                   '16' => '16', '17' => '17', '18' => '18', '19' => '19',
                   '20' => '20', '21' => '21', '22' => '22', '23' => '23'};
    } else {
        $values = [-1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11];
        $labels = {'-1' => $args{'None'},
                    '0' => '12',  '1' => '1',  '2' => '2',   '3' => '3',
                    '4' => '4',   '5' => '5',  '6' => '6',   '7' => '7',
                    '8' => '8',   '9' => '9', '10' => '10', '11' => '11'};

        @radio = radio_group ('-name'    => $args{nameBase} . 'Radio',
                              '-values'  => ['AM', 'PM'],
                              '-default' => $args{default} > 11 ? 'PM' : 'AM',
                              '-onClick' => "amPmRadios (this)");
        $args{default} -= 12 if $args{default} > 11;
    }

    $popup = popup_menu ('-name'     => $args{nameBase} . 'Popup',
                         '-default'  => $args{default},
                         '-values'   => $values,
                         '-labels'   => $labels,
                         '-onChange' => "setMinutesPopup (this)");

    ($popup, @radio);
}

# Produce a popup to select a minute.
# Pash hash pairs with 'name', 'default' keys.
# Defaults to none.
sub _minutePopup {
    my $className = shift;
    my %args = (name         => 'minutePopup',
                default      => -1,
                @_);

    $args{default} = -1 if ($args{default} < -1 or $args{default} > 59);

    my @values = (-1);
    my %labels = (-1 => ' ', -2 => '---');

    for (my $i=0; $i<60; $i+=5) {
        push @values, $i;
        $labels{$i} = ":$i";
    }
    push @values, -2;
    for (my $i=1; $i<60; $i++) {
        push @values, $i;
        $labels{$i} = ":$i";
    }
    for (my $i=0; $i<10; $i++) {
        $labels{$i} = ":0$i";
    }

    popup_menu ('-name'    => $args{name},
                '-default' => $args{default},
                '-values'  => \@values,
                '-labels'  => \%labels);
}

sub _setMinutesPopup {
    my $class = shift;
    my $code = <<END_SCRIPT;
 <SCRIPT LANGUAGE="JavaScript">
 <!-- start
     // Set the Minutes Popup to :00 if the hours were edited to non-blank
     // and the minutes are blank, or to blank if the hours were set to blank
     function setMinutesPopup (hourPopup) {
         form = hourPopup.form
         if (hourPopup.name == 'StartHourPopup') {
             minutePopup = form.StartMinutePopup;
         } else {
             minutePopup = form.EndMinutePopup;
         }
         // If we select None for hour, set minutes to None
         if (hourPopup.selectedIndex < 1) {
             minutePopup.selectedIndex = 0
         }
         // If we select a valid hour and minutes set to none, set them to 0
         if ((hourPopup.selectedIndex > 0) &&
             (minutePopup.selectedIndex < 1)) {
             minutePopup.selectedIndex = 1
         }
     }

     // Set the End Time radio to PM if the start time is PM
     function amPmRadios (radio) {
         if (radio.name == 'StartHourRadio') {
             if (radio.value == 'PM') {
                 radio.form.EndHourRadio[1].checked = true;
             }
         }
     }
 // End -->
 </SCRIPT>
END_SCRIPT

    $code;
}

sub _setTimesFromPeriodJS {
    my ($class, $milTime, $offset, @periods) = @_;

    my @startHours = 0;
    my @startMins  = 0;
    my @startAM    = 0;
    my @endHours   = 0;
    my @endMins    = 0;
    my @endAM      = 0;

    foreach (@periods) {
        next unless ref $_;

        my $startHour = int ($_->[1] / 60) + $offset;
        my $startMin  = $_->[1] % 60;

        my $endHour = int ($_->[2] / 60) + $offset;
        my $endMin  = $_->[2] % 60;


        if ($milTime) {
            push @startHours, $startHour + 1;
            push @endHours,   $endHour + 1;
        } else {
            my $ampm = '"AM"';
            if ($startHour > 12) {
                $startHour -= 12;
                $ampm = '"PM"';
            }
            $startHour = 0 if ($startHour == 12);
            push @startHours, $startHour + 1;
            push @startAM, $ampm;


            $ampm = '"AM"';
            if ($endHour > 12) {
                $endHour -= 12;
                $ampm = '"PM"';
            }
            $endHour = 0 if ($endHour == 12);
            push @endHours, $endHour + 1;
            push @endAM, $ampm;
        }

        if ($startMin == 0) {
            push @startMins, 1;
        } else {
            push @startMins, $startMin + 13;
        }
        if ($endMin == 0) {
            push @endMins, 1;
        } else {
            push @endMins, $endMin + 13;
        }
    }
    my $startHours = join (',', @startHours);
    my $startMins  = join (',', @startMins);
    my $startAM    = join (',', @startAM);

    my $endHours   = join (',', @endHours);
    my $endMins    = join (',', @endMins);
    my $endAM      = join (',', @endAM);

    my @ampmCode = ('','');
    if (!$milTime) {
        @ampmCode =
                (qq /startAM    = [$startAM];
                     endAM      = [$endAM];/,
                 qq /if (startAM[index] == 'AM') {
                         form.StartHourRadio[0].checked = true;
                     } else {
                         form.StartHourRadio[1].checked = true;
                     }

                     if (endAM[index] == 'AM') {
                         form.EndHourRadio[0].checked = true;
                     } else {
                         form.EndHourRadio[1].checked = true;
                     }/);
    }

    my $code = <<END_SCRIPT;
 <SCRIPT LANGUAGE="JavaScript">
 <!-- start
     startHours = [$startHours];
     startMins  = [$startMins];

     endHours   = [$endHours];
     endMins    = [$endMins];
     $ampmCode[0]

     // Set start/end times on period change
     function setTimeFromPeriod (periodPopup) {
         form = periodPopup.form
         index = periodPopup.selectedIndex;

         form.StartHourPopup.selectedIndex = startHours[index];
         form.StartMinutePopup.selectedIndex = startMins[index];

         form.EndHourPopup.selectedIndex = endHours[index];
         form.EndMinutePopup.selectedIndex = endMins[index];

         $ampmCode[1]
     }
 // End -->
 </SCRIPT>
END_SCRIPT

    $code;
}

sub _makeCatPopup {
    my %args = @_;
    my $name       = $args{name};
    my $catObjs    = $args{cats};
    my $selecteds  = $args{selected};
    my $onClick    = $args{onClick};
    my $isMultiple = $args{isMultiple};

    $selecteds ||= [];
    my $style = '';
    if (!$isMultiple and $selecteds->[0] and
        (my $c = $catObjs->{$selecteds->[0]})) {
        my ($fg, $bg) = ($c->fg, $c->bg);
        $style = "style=\"color: $fg; background-color: $bg\"";
    }
    my $mult  = $isMultiple ? 'Multiple'              : '';
    my $click = $onClick    ? qq (onClick="$onClick") : '';
    my $size  = $isMultiple ? "size=$isMultiple"      : '';
    my $catPopup = "<select $style name=$name $mult $size $click>";

    my %sel;
    foreach (@$selecteds) {
        next unless defined;
        $sel{$_} = 1;
    }

    $catPopup .= '<option value="-">-</option>';

    my (@jsBG, @jsFG);
    foreach $name (sort {lc ($a) cmp lc ($b)} keys %$catObjs) {
        $style = '';
        my ($fg, $bg) = ('black', 'lightgray');
        if ($catObjs->{$name}) {
            $fg = $catObjs->{$name}->fg || 'black';
            $bg = $catObjs->{$name}->bg || 'lightgray';
            $style = "style=\"color: $fg; background-color: $bg\"";
        }
        push @jsBG, "'$bg'";
        push @jsFG, "'$fg'";
        my $def = $sel{$name} ? 'selected' : '';
        my $esc = CGI::escapeHTML ($name);
        $catPopup .= "<option $def $style value=\"$esc\">$esc</option>\n";
    }
    $catPopup .= '</select>';
    return ($catPopup, \@jsBG, \@jsFG);
}

1;
